#version 130

//Shader by LoLip_p

#include "/settings.glsl"

in vec2 TexCoords;

uniform float viewWidth, viewHeight;
uniform sampler2D colortex0;
uniform float frameTimeCounter;
uniform float isSneaking;

float rand(vec2 n) {
    return fract(sin(dot(n, vec2(12.9898,78.233))) * 43758.5453);
}
float noise(vec2 x) {
    vec2 i = floor(x);
    vec2 f = fract(x);
    vec2 u = f * f * (3.0 - 2.0 * f);
    float a = rand(i + vec2(0.0,0.0));
    float b = rand(i + vec2(1.0,0.0));
    float c = rand(i + vec2(0.0,1.0));
    float d = rand(i + vec2(1.0,1.0));
    return mix(mix(a, b, u.x), mix(c, d, u.x), u.y);
}

vec2 cameraShake(float t, float intensity, float freq) {
    float nx = noise(vec2(0.0, t * freq));
    float ny = noise(vec2(1.0, t * freq));
    return (vec2(nx, ny) * 2.0 - 1.0) * intensity;
}

vec2 rotateUV(vec2 uv, float angle) {
    vec2 center = vec2(0.5, 0.5);
    vec2 d = uv - center;
    float c = cos(angle);
    float s = sin(angle);
    return center + vec2(
        d.x * c - d.y * s,
        d.x * s + d.y * c
    );
}

#if LENS_FLARE_ENABLE == 1
in float sunVisibility;

in vec2 lf1Pos;
in vec2 lf2Pos;
in vec2 lf3Pos;
in vec2 lf4Pos;
in vec2 lf5Pos;
in vec2 lf6Pos;
in vec2 lf7Pos;
in vec2 lf8Pos;
in vec2 lf9Pos;
in vec2 lf10Pos;
in vec2 lf11Pos;
in vec2 lf12Pos;
in vec2 lf13Pos;

uniform float rainStrength;
uniform float aspectRatio;

#define LENS_FLARE(COLOR, UV, LFPOS, LFSIZE, LFCOLOR) { \
                vec2 delta = UV - LFPOS; delta.x *= aspectRatio; \
                if(abs(delta.x)+abs(delta.y) < LFSIZE * 2.0) { \
                    float d = max(LFSIZE - sqrt(dot(delta, delta)), 0.0); \
                    COLOR += LFCOLOR.rgb * (LFCOLOR.a * 0.5) * smoothstep(0.0, LFSIZE, d) * sunVisibility;\
                } }

vec3 lensFlare(vec3 color, vec2 uv) {
    if(sunVisibility <= 0.0 || rainStrength >= 0.3)
    return color;
    LENS_FLARE(color, uv, lf1Pos, 0.4, vec4(1.0, 1.0, 0.9, 0.15));
    LENS_FLARE(color, uv, lf2Pos, 0.6, vec4(0.9, 1.0, 0.7, 0.15));
    LENS_FLARE(color, uv, lf3Pos, 0.5, vec4(1.0, 0.7, 0.0, 0.15));
    LENS_FLARE(color, uv, lf4Pos, 0.7, vec4(0.9, 0.1, 0.0, 0.15));
    LENS_FLARE(color, uv, lf5Pos, 0.4, vec4(0.2, 0.06, 0.6, 0.4));
    LENS_FLARE(color, uv, lf6Pos, 0.5, vec4(0.4, 0.03, 1.0, 0.2));
    LENS_FLARE(color, uv, lf7Pos, 0.02, vec4(0.5, 0.85, 0.85, 0.5));
    LENS_FLARE(color, uv, lf8Pos, 0.03, vec4(0.5, 0.5, 0.9, 0.5));
    LENS_FLARE(color, uv, lf9Pos, 0.03, vec4(0.3, 0.4, 1.0, 0.5));
    LENS_FLARE(color, uv, lf10Pos, 0.05, vec4(1.5, 0.85, 0.45, 0.3));
    LENS_FLARE(color, uv, lf11Pos, 0.01, vec4(1.3, 0.8, 0.4, 0.1));
    LENS_FLARE(color, uv, lf12Pos, 0.1, vec4(1.3, 1.0, 1.0, 0.4));
    LENS_FLARE(color, uv, lf13Pos, 0.1, vec4(1.0, 1.0, 1.0, 0.1));
    return color;
}
#endif

void main() {
    vec3 color;

    // Apply zoom
    vec2 centeredCoord = (TexCoords - 0.5) * ZOOM + 0.5;
    float distance = length(centeredCoord - 0.5);

    // Camera shake effect
	vec2 shake = cameraShake(frameTimeCounter, INTENSITY_CAM_SHAKE, 3.0f);

    // Fisheye effect
    vec2 fisheyeUV = (centeredCoord + shake - 0.5) * (1.0 + DIST_STRENGTH * distance * distance) + 0.5;

	// Chromatic aberration
    vec2 redUV = fisheyeUV + vec2(CHROMATIC_ABERRATION, CHROMATIC_ABERRATION) * distance;
    vec2 blueUV = fisheyeUV + vec2(-CHROMATIC_ABERRATION, -CHROMATIC_ABERRATION) * distance;
    color.r = texture2D(colortex0, redUV).r;
    color.g = texture2D(colortex0, fisheyeUV).g;
    color.b = texture2D(colortex0, blueUV).b;

    // Sharpening
    vec2 texelSize = 1.0 / textureSize(colortex0, 0);
    vec3 sharpenedColor = vec3(0.0);

    // Define the sharpening kernel
    sharpenedColor += texture2D(colortex0, fisheyeUV + texelSize * vec2(-1.0,  0.0)).rgb * -1.0;
    sharpenedColor += texture2D(colortex0, fisheyeUV + texelSize * vec2( 1.0,  0.0)).rgb * -1.0;
    sharpenedColor += texture2D(colortex0, fisheyeUV + texelSize * vec2( 0.0, -1.0)).rgb * -1.0;
    sharpenedColor += texture2D(colortex0, fisheyeUV + texelSize * vec2( 0.0,  1.0)).rgb * -1.0;
    sharpenedColor += texture2D(colortex0, fisheyeUV).rgb * 5.0;

    // Combine the sharpened color with the original color
    color = mix(color, sharpenedColor, SHARPNESS_STRENGTH);
	color = clamp(color, 0, 1);
	
	#if LENS_FLARE_ENABLE == 1
    color = lensFlare(color, fisheyeUV);
	#endif
	
	color = pow(color, vec3(0.45f));
	
    #if SCANLINE == 1
	float scanline = sin(fisheyeUV.y * SCANLINE_WIDTH * 1.5) * SCANLINE_STRENGTH;
    color.rgb += scanline;
    #endif
	
    #if GRAIN == 1
    float noise = (fract(sin(dot(fisheyeUV, vec2(12.9898, 78.233 * frameTimeCounter))) * 43758.5453) - 0.5) * GRAIN_STRENGTH;
    color.rgb += vec3(noise);
    #endif

    #if COLOR_DIST == 1
    float colorDistort = COLOR_DIST_STRENGTH * sin(frameTimeCounter * 2.0);
    color.rgb *= vec3(1.0 + colorDistort, 1.0 - colorDistort, 1.0 + colorDistort);
    #endif


	// NVG
	#if ENABLE_NVG_IsSneaking == 1
	if(isSneaking == 1.0)
	{
		float gray = dot(color.rgb, vec3(0.299, 0.587, 0.114));
		vec3 grayscale = vec3(gray);
		vec3 colorTransform = vec3(R, G, B) * degree_brightness_increase;
		color = grayscale * colorTransform;
	}
	
	#else
	#if NVG == 1
    float gray = dot(color.rgb, vec3(0.299, 0.587, 0.114));
    vec3 grayscale = vec3(gray);
    vec3 colorTransform = vec3(R, G, B) * degree_brightness_increase;
    color = grayscale * colorTransform;
	#endif
	
	#endif

    // Apply saturation, brightness, and contrast
    float average = (color.r + color.g + color.b) / 3.0;
    color.rgb = mix(vec3(average), color.rgb, SATURATION);
    color.rgb += BRIGHTNESS;
    color.rgb = ((color.rgb - 0.5) * CONTRAST) + 0.5;
	
    #if VIGNETTE == 1
    float vignette = smoothstep(VIGNETTE_RADIUS, VIGNETTE_RADIUS + VIGNETTE_STRENGTH, distance);
    color = mix(color, vec3(0.0), vignette);
    #endif

    #if BLACK_STRIPES == 1
    float leftStripe = smoothstep(BLACK_STRIPES_WIDTH, BLACK_STRIPES_WIDTH - BLACK_STRIPES_SOFT, TexCoords.x);
    float rightStripe = smoothstep(1.0 - BLACK_STRIPES_WIDTH, 1.0 - (BLACK_STRIPES_WIDTH - BLACK_STRIPES_SOFT), TexCoords.x);
    float stripeEffect = max(leftStripe, rightStripe);
    color = mix(color, vec3(0.0), stripeEffect);
	#endif

    gl_FragColor = vec4(color, 1.0);
}
