#version 130

// Shader by LoLip_p

#include "distort.glsl"
#include "/settings.glsl"

in vec2 TexCoords;

uniform vec3 sunPosition;
uniform float rainStrength;
uniform int worldTime;
uniform vec3 playerPosition;

uniform sampler2D colortex0;
uniform sampler2D colortex2;
uniform sampler2D colortex3;
uniform sampler2D colortex4;
uniform sampler2D depthtex0;
uniform sampler2D depthtex1;

// For Shadow
uniform sampler2D shadowtex0;
uniform sampler2D shadowtex1;
uniform sampler2D shadowcolor0;
uniform mat4 gbufferProjectionInverse;
uniform mat4 gbufferModelViewInverse;
uniform mat4 shadowModelView;
uniform mat4 shadowProjection;

uniform float frameTimeCounter;

#if LUT == 0 && BIT == 1
/*
const int colortex0Format = R11F_G11F_B10F;
*/
#else
/*
const int colortex0Format = RGB8;
*/
#endif

/*
const int colortex2Format = RGB10_A2;
const int colortex3Format = RG16;
const int colortex4Format = RGBA2;
const int colortex5Format = R8;
const int colortex6Format = RGB8;
const int colortex7Format = R8;
*/

const int noiseTextureResolution = 32;

const vec3 MOD3 = vec3(.1031,.11369,.13787);
float hash12(vec2 p)
{
	vec3 p3  = fract(vec3(p.xyx) * MOD3);
    p3 += dot(p3, p3.yzx + 19.19);
    return fract((p3.x + p3.y) * p3.z);
}

float calculateShadowBias(float distance) {
    return mix(0.0001f, 0.003f, clamp(distance / shadowDistance, 0.0f, 1.0f));
}

float Visibility(in sampler2D ShadowMap, in vec3 SampleCoords, in vec3 WorldPosition) {
	float distance = length(WorldPosition - playerPosition);
    float bias = calculateShadowBias(distance);
    return step(SampleCoords.z - bias, texture2D(ShadowMap, SampleCoords.xy).r);
}

vec3 TransparentShadow(in vec3 SampleCoords, in vec3 WorldPosition){
    float ShadowVisibility0 = Visibility(shadowtex0, SampleCoords, WorldPosition);
    float ShadowVisibility1 = Visibility(shadowtex1, SampleCoords, WorldPosition);
    vec4 ShadowColor0 = texture2D(shadowcolor0, SampleCoords.xy);
    vec3 TransmittedColor = ShadowColor0.rgb * (1.0f - ShadowColor0.a);
    return mix(TransmittedColor * ShadowVisibility1, vec3(1.0f), ShadowVisibility0);
}

const int ShadowSamplesPerSize = 2 * SOFT + 1;
const int TotalSamples = ShadowSamplesPerSize * ShadowSamplesPerSize;

vec3 GetShadow(float depth) {
    vec3 ClipSpace = vec3(TexCoords, depth) * 2.0f - 1.0f;
	
    vec4 ViewW = gbufferProjectionInverse * vec4(ClipSpace, 1.0f);
    vec3 View = ViewW.xyz / ViewW.w;
	
    vec4 World = gbufferModelViewInverse * vec4(View, 1.0f);
    vec4 ShadowSpace = shadowProjection * shadowModelView * World;
    ShadowSpace.xyz = DistortPosition(ShadowSpace.xyz);
    vec3 SampleCoords = ShadowSpace.xyz * 0.5f + 0.5f;
	
	SampleCoords = clamp(SampleCoords, 0.0, 1.0); // clamp
	
    float RandomAngle = hash12((TexCoords + frameTimeCounter) * 100);
    float cosTheta = cos(RandomAngle);
	float sinTheta = sin(RandomAngle);
    mat2 Rotation =  mat2(cosTheta, -sinTheta, sinTheta, cosTheta) / shadowMapResolution; // We can move our division by the shadow map resolution here for a small speedup
    vec3 ShadowAccum = vec3(0.0f);
    for(int x = -SOFT; x <= SOFT; x++){
        for(int y = -SOFT; y <= SOFT; y++){
            vec2 Offset = Rotation * vec2(x, y);
            vec3 CurrentSampleCoordinate = vec3(SampleCoords.xy + Offset, SampleCoords.z);
			
			CurrentSampleCoordinate = clamp(CurrentSampleCoordinate, 0.0, 1.0); // clamp
			
            ShadowAccum += TransparentShadow(CurrentSampleCoordinate, World.xyz);
        }
    }
    ShadowAccum /= TotalSamples;
	
	float distanceShadowFade = length(View) / shadowDistance;
	float shadowFade = smoothstep(0.0, 1.0, 1.0 - distanceShadowFade);

	ShadowAccum = clamp(mix(vec3(1), ShadowAccum, 1 - pow(1 - shadowFade, 10)), 0, 1);
	
    return ShadowAccum;
}

float AdjustLightmapTorch(in float torchLight) {
    const float K = 2.0f;
	const float P = 4.0f;
    return K * pow(torchLight, P);
}

float AdjustLightmapSky(in float sky){
    float sky_2 = sky * sky;
    return sky_2 * sky_2;
}

vec2 AdjustLightmap(in vec2 Lightmap){
    vec2 NewLightMap;
    NewLightMap.x = AdjustLightmapTorch(Lightmap.x);
    NewLightMap.y = AdjustLightmapSky(Lightmap.y);
    return NewLightMap;
}

vec3 GetLightmapColor(in vec2 Lightmap, float state){
    Lightmap = AdjustLightmap(Lightmap);
	
	#if TI == 0
    const vec3 SkyColor = vec3(0.05f, 0.15f, 0.3f);
    const vec3 TorchColor = vec3(1.0f, 0.85f, 0.7f);
	#else
	const vec3 SkyColor = vec3(0);
	const vec3 TorchColor = vec3(1);
	#endif
	
	
    vec3 TorchLighting = Lightmap.x * vec3(1.0f, 1.0f, 1.0f);
    vec3 SkyLighting = Lightmap.y * SkyColor * state * (1 - rainStrength);
	
	#if TI == 0
	//TorchLighting -= (Lightmap.y * LIGHT_ABSORPTION * state * (1 - rainStrength));
	#endif
	
	float occ = (Lightmap.y * LIGHT_ABSORPTION * state * (1 - rainStrength));
	float torchMask = smoothstep(-0.2, 0.8, Lightmap.x);
	TorchLighting -= occ * torchMask;
	
	vec3 LightmapLighting = clamp(TorchLighting * TorchColor, 0.0f, 1.0f) + SkyLighting;
	
    return LightmapLighting;
}

float smoothTransition(float time) {
    if (time >= 0.0f && time <= 1000.0f) {
        return time / 1000.0f; // Transition from 0 to 1
    } else if (time > 1000.0f && time < 12000.0f) {
        return 1.0f; // Fully enabled
    } else if (time >= 12000.0f && time <= 13000.0f) {
        return 1.0f - (time - 12000.0f) / 1000.0f; // Transition from 1 to 0
    } else {
        return 0.0f; // Fully disabled
    }
}

float swapDayNight(float time) {
    if (time >= 12300.0f && time <= 12800.0f) {
        return (time - 12300.0f) / 500.0f; // Плавный переход от 0 до 1
    } else if (time > 12800.0f && time <= 13200.0f) {
        return 1.0f - (time - 12800.0f) / 400.0f; // Плавный переход от 1 до 0
    } else if (time >= 22700.0f && time <= 23200.0f) {
        return (time - 22700.0f) / 500.0f; // Плавный переход от 0 до 1
    } else if (time > 23200.0f && time <= 23700.0f) {
        return 1.0f - (time - 23200.0f) / 500.0f; // Плавный переход от 1 до 0
    } else {
        return 0.0f; // Вне указанных диапазонов
    }
}

#if AO == 1
const float BIAS = 0.05;

vec3 getPosition(vec2 uv) {
    float z = texture2D(depthtex1, uv).r;
	vec2 ndc = uv * 2.0 - 1.0;
	vec4 clip = vec4(ndc, z * 2.0 - 1.0, 1.0);
	vec4 view = gbufferProjectionInverse * clip;
	view /= view.w;
	return view.xyz;
}

vec3 getNormal(vec2 uv) {
    vec3 n = texture2D(colortex4, uv).xyz * 2.0 - 1.0;
    return normalize(n);
}

// Sample one AO ray
float doAO(vec2 uv, vec2 dir, vec3 p, vec3 n) {
    vec3 samplePos = getPosition(uv + dir) - p;
    float dist = length(samplePos);
    vec3  v    = samplePos / dist;
    float d    = dist * SCALE;
    float ao   = max(dot(n, v) - BIAS, 0.0) / (1.0 + d);
    ao *= smoothstep(MAX_DISTANCE, MAX_DISTANCE * 0.5, dist);
    return ao;
}

// Spiral-pattern AO
float spiralAO(vec2 uv, vec3 p, vec3 n, float rad) {
    const float golden = 2.4; // ≈ π*(3 - √5)
    float ao = 0.0;
    float inv = 1.0 / float(SAMPLES);
	float radius = 0.0;
    float phase = hash12((uv + frameTimeCounter) * 100.0) * 6.28;
    float step = rad * inv;
	
    for (int i = 0; i < SAMPLES; i++) {
        vec2 dir = vec2(sin(phase), cos(phase));
		radius += step;
        phase += golden;
        ao += doAO(uv, dir * radius, p, n);
    }
    return ao * inv;
}
#endif

uniform int isEyeInWater;
uniform float near, far;
uniform float blindness;
uniform float darknessFactor;
uniform vec3 fogColor;

vec3 projectAndDivide(mat4 projectionMatrix, vec3 position){
	vec4 homPos = projectionMatrix * vec4(position, 1.0);
	return homPos.xyz / homPos.w;
}

void main(){
	vec3 Albedo = pow(texture2D(colortex0, TexCoords).rgb, vec3(2.2f));
	float state = smoothTransition(worldTime);
	
	float Depth = texture2D(depthtex0, TexCoords).r;
    if(Depth == 1.0f){
	    vec3 colorSky = Albedo;
		
		#if TI == 1
	    colorSky = mix(vec3(0.18), vec3(0.0005), (1.0f - state));
		#endif
	
		/* DRAWBUFFERS:07 */
        gl_FragData[0].rgb = colorSky;
        gl_FragData[1].r = 1.0f;
        return;
    }
	
	float Depthv2 = texture2D(depthtex1, TexCoords).r;
	float normal = texture2D(colortex3, TexCoords).x;
	
	vec3 lightBrightnessV = vec3(0.0);
    vec3 ShadowColor = vec3(1);
	
	vec3 Lightmap = texture2D(colortex2, TexCoords).xyz;
	vec3 LightmapColor = GetLightmapColor(Lightmap.xy, state);
	
	float ao = 1.0;
	
	#if TI == 0
	#if SHADING == 1
    float lightBrightness = Lightmap.z;
	
	lightBrightness *= state;
	lightBrightness *= 1.0f - rainStrength;
	lightBrightness += (rainStrength - 0.5f) * (1.0f - state) * rainStrength; //Rain
	lightBrightnessV = vec3(lightBrightness) * state;
	lightBrightnessV += vec3(0.2, 0.2, 0.5) * (1.0f - state); //Night
	#endif
	
	lightBrightnessV *= AdjustLightmap(Lightmap.xy).g;
	
	#if FOG == 1
	vec3 NDCPos = vec3(TexCoords.xy, Depth) * 2.0 - 1.0;
	vec3 viewPos = projectAndDivide(gbufferProjectionInverse, NDCPos);
	
	float farW = far;
	float distW = FOG_DISTANCE;
	
	if(isEyeInWater == 1) {
		farW = 16;
		distW = 1;
	}
	
	float distance = length(viewPos) / (farW * mix(mix(1, near * 0.5, blindness), near, darknessFactor));
	float fogFactor = exp(-mix(distW, 1, blindness) * (1.0 - distance));
	
	vec3 color_sky;
	#if TI == 0
	
	#if GREY_WEATHER == 0
	color_sky = fogColor;
	#else
	color_sky = mix(fogColor, vec3(0.75, 0.75, 0.76), state);
	#endif
	
	#else
	color_sky = pow(mix(vec3(0.0005), vec3(0.18), state), vec3(1.0f / 2.2f));
	#endif
	
	if(Depthv2 == 1 && normal == 0) {
		/*DRAWBUFFERS:07*/
		gl_FragData[0].rgb = Albedo;
		gl_FragData[1].r = 1.0f;
		return;
	}
	#endif
	
	if(normal > 0.002f) {
		#if ENABLE_SHADOW == 1
		vec3 originalShadow = GetShadow(Depth);
		float shadow_transperency_coef = mix(degree_night_darkness, SHADOW_TRANSPARENCY, state);
		ShadowColor = mix(originalShadow * vec3(shadow_transperency_coef / 0.05), originalShadow, state);
		ShadowColor = mix(ShadowColor, vec3(1), swapDayNight(worldTime));
		vec3 colorRainDayOrNight = mix(vec3(0.2), vec3(1), smoothTransition(worldTime));
		ShadowColor = mix(ShadowColor, colorRainDayOrNight, rainStrength);
	
		ShadowColor *= AdjustLightmap(Lightmap.xy).g;
		ShadowColor += (SHADOW_TRANSPARENCY / 0.1) * (1 - state);
		#endif
		
		LightmapColor *= mix(5, 1, state);
		
		Albedo *= clamp(state, SHADOW_TRANSPARENCY + 0.05, 1.0f);
		Albedo *= (LightmapColor + lightBrightnessV * ShadowColor);
		
		#if FOG == 1
		Albedo = mix(Albedo, pow(color_sky, vec3(2.2f)), clamp(fogFactor, 0.0, 1.0));
		#endif
		
		/* DRAWBUFFERS:07 */
        gl_FragData[0].rgb = Albedo;
        gl_FragData[1].r = 1.0f;
        return;
    }
	
	#if AO == 1
	vec2 ndc = TexCoords * 2.0 - 1.0;
	vec4 clip = vec4(ndc, Depth * 2.0 - 1.0, 1.0);
	vec4 view = gbufferProjectionInverse * clip;
	view /= view.w;
	
	float distanceFade = clamp(length(view.xyz) / (far / 1.5), 0.0, 1.0);
	float fade = pow(smoothstep(0.0, 1.0, 1.0 - distanceFade), 0.25);
	
	if (fade > 0.15) { // Отсекаем очень дальние пиксели
		vec3 p = getPosition(TexCoords);
		vec3 n = getNormal(TexCoords);
		float rad = SAMPLE_RAD / abs(p.z);
		
		float rawAO = spiralAO(TexCoords, p, n, rad);
		float aoStrength = 1.0 - rawAO * INTENSITY;
		ao = mix(1.0, aoStrength, fade); // плавное применение AO
	}
	#endif
	
	float shadow_transperency_coef = mix(degree_night_darkness, SHADOW_TRANSPARENCY, state);

    #if ENABLE_SHADOW == 1
	if (float(Depth < 0.56) < 0.5) {
		vec3 originalShadow = GetShadow(texture2D(depthtex1, TexCoords).r);
		ShadowColor = mix(originalShadow * vec3(shadow_transperency_coef / 0.75), originalShadow, smoothTransition(worldTime));
		ShadowColor = mix(ShadowColor, vec3(1), swapDayNight(worldTime));
		vec3 colorRainDayOrNight = mix(vec3(0.2), vec3(1), state);
		ShadowColor = mix(ShadowColor, colorRainDayOrNight, rainStrength);
		
		ShadowColor *= AdjustLightmap(Lightmap.xy).g;
		//ShadowColor += shadow_transperency_coef;
	}
	else
	{
		ao = 1;
	}
	#endif
	
	ShadowColor *= mix(vec3(1.0f), lightBrightnessV, state);
	ShadowColor += shadow_transperency_coef;
	ShadowColor *= mix(lightBrightnessV, vec3(1.0f), state);

	vec3 Diffuse = Albedo * (LightmapColor + ShadowColor);
	
	#if FOG == 1
	Diffuse = mix(Diffuse, pow(color_sky, vec3(2.2f)), clamp(fogFactor, 0.0, 1.0));
	#endif
	
	/* DRAWBUFFERS:07 */
    gl_FragData[0].rgb = clamp(Diffuse, 0, 1);
    gl_FragData[1].r = ao;
	
	#else
	float lightBrightness = Lightmap.z;
	
	lightBrightness *= state;
	lightBrightness *= 1.0f - rainStrength;
	lightBrightness += (rainStrength - 0.5f) * (1.0f - state) * rainStrength; //Rain
	lightBrightnessV = vec3(lightBrightness) * state;
	
	lightBrightnessV *= AdjustLightmap(Lightmap.xy).g;
	lightBrightnessV += 0.1;
	
	/* DRAWBUFFERS:07 */
    gl_FragData[0].rgb = LightmapColor + lightBrightnessV;
    gl_FragData[1].r = ao;
	#endif
}
